<?php
namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;


use App\Http\Resources\HousingUnitPlotResource;
use App\Models\HousingUnitPlot;
use App\Models\HousingUnitPremium;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use App\Traits\ApiResponseTrait;
use Illuminate\Support\Facades\DB;


class HousingUnitPlotController extends Controller
{

    
    use ApiResponseTrait;
    protected function store(Request $request)
    {
        if (count($request->all()) <= 0) {
            return response()->json([
                'message' => 'No record to insert',
            ], 404);
            dd($request);

        }
        $stored = HousingUnitPlot::insert($request->all());
        return $stored ? $this->successInsertResponse($stored) : $this->errorInsertResponse();
    }

   
    protected function index(Request $request, $id = null)
    {
        $serverId[] = $request->server_id;
        $limit = $request->input('limit', DEFAULT_LIMIT);

        $data = HousingUnitPlot::whereId($id)
            ->whereServerId($serverId)
            ->whereUpdateAble($request->updated_status)
            ->Where('Comp_ID', COMPANY_ID)
            ->paginate($limit);


        return $data ? $this->recordFoundResponse($data) : $this->recordNotFoundResponse();
    }

    protected function update(Request $request)
    {
        $record = HousingUnitPlot::find($request->id);
        if (!$record) {
            return $this->generalResponse('RECORD_NOT_FOUND', 404);
        }
        $updatedStatus = [
            'updated_status' => '1',
        ];
        $request->merge($updatedStatus);
        $record->update($request->all());
        return $record ? $this->successUpdateResponse($record) : $this->errorUpdateResponse();
    }


    // api for check id's of sql server
    public function serverId()
    {
        $data = HousingUnitPlot::select('server_id as id')->Where('Comp_ID', COMPANY_ID)->get();
        return $data->isNotEmpty() ? $this->successResponse($data, RECORD_FOUND, HTTP_OK) : $this->generalResponse(RECORD_NOT_FOUND, HTTP_NOT_FOUND);
    }


    public function changeUpdatedStatus(Request $request)
    {
        $serverIds = $request->server_id;
        $execute = DB::table('Housing_UnitPlot_Tab')
            ->whereIn('server_id', $serverIds)
            ->update(['updated_status' => '0']);
        return $execute ? $this->successUpdateResponse($serverIds) : $this->errorUpdateResponse();
    }


    public function editBySql(Request $request)
    {
        $successCount = 0;
        $totalCount = count($request->data);

        foreach ($request->data as $list) {
            // return $list['server_id'];
            $result = DB::table('Housing_UnitPlot_Tab')
                ->where('server_id', $list['server_id'])
                ->update([
                    'UP_ID' => $list['UP_ID'],
                    'Comp_ID' => $list['Comp_ID'],
                    'Project_ID' => $list['Project_ID'],
                    'HUC_ID' => $list['HUC_ID'],
                    'HUSC_ID' => $list['HUSC_ID'],
                    'HUP_ID' => $list['HUP_ID'],
                    'HUSL_ID' => $list['HUSL_ID'],
                    'HUB_ID' => $list['HUB_ID'],
                    'price' => $list['price'],
                    'HUS_ID' => $list['HUS_ID'],
                    'OS_LID' => $list['OS_LID'],
                    'UD_F' => $list['UD_F'],
                    'UD_B' => $list['UD_B'],
                    'UD_L' => $list['UD_L'],
                    'UD_R' => $list['UD_R'],
                    'U_W' => $list['U_W'],
                    'U_L' => $list['U_L'],
                    'UP_Marla' => $list['UP_Marla'],
                    'UPSys_Marla' => $list['UPSys_Marla'],
                    'UP_Status' => $list['UP_Status'],
                    'Size_Type' => $list['Size_Type'],
                    'UP_Type' => $list['UP_Type'],
                    'Sales_UP_ID' => $list['Sales_UP_ID'],
                    'CGS_UP_ID' => $list['CGS_UP_ID'],
                    'WIP_UP_ID' => $list['WIP_UP_ID'],
                    'Covered_Area' => $list['Covered_Area'],
                    'HCS_ID' => $list['HCS_ID'],
                ]);

            if ($result !== false) {
                $successCount++;
            }
        }

        return $successCount === $totalCount ? $this->successUpdateResponse("All records updated successfully.") : $this->errorUpdateResponse("Failed to update some records.");

    }



    public function comparePlot(Request $request, $id = null)
    {

        $limit = $this->setDefaultValue($request->limit, PHP_INT_MAX);
        $minPrice = $this->setDefaultValue($request->min_price);
        $maxPrice = $this->setDefaultValue($request->max_price);
        $marla = $this->setDefaultValue($request->marla);
        $propertyType = $this->setDefaultValue($request->property_type);
        $width = $this->setDefaultValue($request->width);
        $length = $this->setDefaultValue($request->length);
        $city = $this->setDefaultValue($request->city);
        $location = $this->setDefaultValue($request->location);
        $category = $this->setDefaultValue($request->category);
        $searchTerm = $this->setDefaultValue($request->search_term);
        $plotList = $this->setDefaultValue($request->plot_list);


        // DB::enableQueryLog();
        $data = HousingUnitPlot::select(
            'Housing_UnitPlot_Tab.ID as id',
            'Housing_Unit_Premium_Tab.HUP_Name',
            'Housing_UnitPlot_Tab.Per_M_Rate as price',
            'Housing_UnitPlot_Tab.UP_ID',
            'Housing_UnitPlot_Tab.UP_Marla',
            'Housing_UnitPlot_Tab.UD_F',
            'Housing_UnitPlot_Tab.UD_B',
            'Housing_UnitPlot_Tab.UD_L',
            'Housing_UnitPlot_Tab.UD_R',
            'Housing_UnitPlot_Tab.U_W',
            'Housing_UnitPlot_Tab.U_L',
            'Housing_Project.p_city',
            'Housing_UnitPlot_Tab.Project_ID',
            'Housing_Project.p_loc','Housing_UnitPlot_Tab.UP_Status as plot_status',
            'Account.Acco_Name as plot_name',

        )
            ->Join('Housing_Unit_Premium_Tab', 'Housing_UnitPlot_Tab.HUP_ID', 'Housing_Unit_Premium_Tab.HUP_ID')
            ->Join('Housing_Project', 'Housing_UnitPlot_Tab.Project_ID', 'Housing_Project.Project_ID')
            ->Join('Account', 'Housing_UnitPlot_Tab.UP_ID', 'Account.Acco_LID')
            ->where('Housing_UnitPlot_Tab.Comp_ID', COMPANY_ID)
            ->whereMin($minPrice)
            ->whereMax($maxPrice)
            ->whereMarla($marla)
            ->wherePropertyType($propertyType)
            ->where('Housing_UnitPlot_Tab.UP_Marla' , '<=' , 20)
            // ->where('Housing_Project.id' , 1)
            ->filterByDimension($width, $length)
            ->whereCity($city)
            ->whereLocation($location)
            ->whereCategory($category)
            ->whereId($id)
            ->search($searchTerm)
            ->wherePlotsIn($plotList)
            ->paginate($limit); 

        // return DB::getQueryLog();

        $pagination = [
            'page' => $data->currentPage(),
            'total_record' => $data->total(),
            'limit' => $limit,
            'total_pages' => ceil($data->total() / $limit),
        ];


        return $data->isEmpty()
            ? response()->json(['message' => 'no record found !'], 404)
            : $this->recordFoundResponse(HousingUnitPlotResource::collection($data->items()), 200, $pagination,);

    }


   

    public function mixMaxValues()
    { 
        $prices = HousingUnitPlot::select(
            DB::raw(
            'MIN(Per_M_Rate * UP_Marla) as min_price,
            MAX(Per_M_Rate * UP_Marla) as max_price,
            MIN(UP_Marla) as min_marla,
            MAX(UP_Marla) as max_marla'
            )
        )->where('UP_Marla','<=', 20)->first();

        return response()->json([
         
            'min_price' => ceil($prices->min_price),
            'max_price' => ceil($prices->max_price),
            'min_marla' => floor(5),
            'max_marla' => ceil(20),
        ], 200);
    }
     //
    public function mixMaxMarla()
    {
        $prices = HousingUnitPlot::select(DB::raw('MIN(UP_Marla) as min_marla, MAX(UP_Marla) as max_marla'))->first();
        return response()->json([
            'min_marla' => ceil($prices->min_marla),
            'max_marla' => ceil($prices->max_marla),
        ], 200);
    }


    public function plotReport(Request $request)
    {
        $plotCounts = HousingUnitPlot::select('UP_Status', \DB::raw('count(*) as count'))
        ->groupBy('UP_Status')
        ->pluck('count', 'UP_Status');
    
        return response()->json([
            'available' => $plotCounts->get('AVAILABLE', 0),
            'sold' => $plotCounts->get('SOLD', 0),
            'token' => $plotCounts->get('TOKEN', 0),
            'total' => ($plotCounts->get('AVAILABLE', 0) + $plotCounts->get('SOLD', 0) + $plotCounts->get('TOKEN', 0))
        ]); 
    
    }
    public function inventoryReport(Request $request)
    {
        $plots = HousingUnitPlot::select(
            'UP_Status',
            'Housing_UnitPlot_Tab.id as plot_id',
            'accounts.Acco_Name as plot_name',
            'Housing_UnitPlot_Tab.up_marla as marla',
            'Housing_UnitPlot_Tab.U_W as width',
            'Housing_UnitPlot_Tab.U_L as length'
        )
        ->join('accounts', 'Housing_UnitPlot_Tab.UP_ID', '=', 'accounts.Acco_LID')
        ->get()
        ->groupBy('UP_Status');
    
        $response = [];
        foreach ($plots as $status => $plotList)
        {
            $response[$status] = [];
    
            foreach ($plotList as $plot)
            {
                $response[$status][] = [
                    'plot_id' => $plot->plot_id,
                    'plot_name' => $plot->plot_name,
                    'marla' => $plot->marla,
                    'width' => $plot->width,
                    'length' => $plot->length,
                ];
            }
        }
    
        return response()->json($response);
    }
    



   

    public function plotList(Request $request, $id = null)
    {
        $data = HousingUnitPremium::select(
            'housing_unit_Premiums.HUP_Name as category',
            'housing_unit_Premiums.HUP_ID'
           )
            ->with([
                'plot' => function ($query) use ($request, $id) {
                    $query->select(
                        'Housing_UnitPlot_Tab.id',
                        // 'categories.huc_name as category',
                        'Housing_UnitPlot_Tab.HUP_ID',
                        'Housing_UnitPlot_Tab.price',
                        'Housing_UnitPlot_Tab.UP_ID as plot_no',
                        'Housing_UnitPlot_Tab.UP_Marla as marla',
                        'Housing_UnitPlot_Tab.UD_F',
                        'Housing_UnitPlot_Tab.UD_B',
                        'Housing_UnitPlot_Tab.UD_L',
                        'Housing_UnitPlot_Tab.UD_R',
                        'Housing_UnitPlot_Tab.U_W as width',
                        'Housing_UnitPlot_Tab.U_L as length',
                        'housing_projects.p_city as city',
                        'Housing_UnitPlot_Tab.Project_ID',
                        'housing_projects.p_loc as location',
                        DB::raw('Housing_UnitPlot_Tab.price * Housing_UnitPlot_Tab.UP_Marla as totalPrice')
                    )
                        // ->leftJoin('categories', 'Housing_UnitPlot_Tab.HUC_ID', 'categories.id')
                        ->leftJoin('housing_projects', 'Housing_UnitPlot_Tab.Project_ID', 'housing_projects.project_id')
                        ->where('Housing_UnitPlot_Tab.UP_Status', 'Available')
                        ->Where('Housing_UnitPlot_Tab.Comp_ID', COMPANY_ID)
                        ->whereMin($request->min_price)
                        ->whereMax($request->max_price)
                        ->whereMarla($request->marla)
                        ->wherePropertyType($request->property_type)
                        ->whereWidth($request->width)
                        ->whereLength($request->length)
                        ->whereCity($request->city)
                        ->whereLocation($request->location)
                        ->whereCategory($request->category)
                        ->whereId($id);
                }
            ])
            ->where('housing_unit_Premiums.B_Group', 'HOUSING')
            ->get();

        if (!$data->isEmpty()) {
            $totalPlots = 0;
            $response = $data->map(function ($item) use (&$totalPlots) {
                $plotNumbers = $item->plot->map(function ($plot) {
                    return $plot->plot_no;
                })->toArray();

                $totalPlots += count($plotNumbers);

                return [
                    'category' => $item->category,
                    'HUP_ID' => $item->HUP_ID,
                    'plots' => $plotNumbers
                ];
            });

            return response()->json([
                'data' => $response,
                'total_plots' => $totalPlots
            ], 200);
        } else {
            return $this->recordNotFoundResponse();
        }
    }



    /**
     * Group the collection date by category
     *
     * @param  collection $data
     * @return array
     */
    private function groupByCategory(Collection $data): array
    {
        $res = [];
        $categories = collect($data)->pluck('category')->unique()->values()->toArray();

        foreach ($categories as $category) {
            $categoryData = collect($data)->map(function ($object) use ($category) {
                if ($object['category'] == $category)
                    return $object;
            })->filter()->values()->toArray();
            return $res[] = [
                'category' => $category,
                'data' => $categoryData,
            ];
        }

    }




}